library(dplyr)

# Function to run simulation and calculate tau_ind
run_simulation <- function() {
  n_listings <- 1000
  time_step <- 0.1 # Time step
  total_time <- 1000 # Total simulation time
  current_time <- 0
  statuses <- rep("available", n_listings)
  lambda_tot <- 10
  
  # Record when each listing will become available again
  availability_times <- rep(0, n_listings)
  
  # Record the number of times each listing has been booked
  order_counts <- rep(0, n_listings)
  
  # Randomly assign 50% of the listings to treatment and the remaining to control
  treatment_listings <- sample(c(rep("treatment", n_listings / 2), rep("control", n_listings / 2)))
  
  # Function to update statuses
  update_statuses <- function(statuses, availability_times, current_time) {
    for (i in 1:length(statuses)) {
      if (statuses[i] == "unavailable" && current_time >= availability_times[i]) {
        statuses[i] <- "available"
      }
    }
    return(statuses)
  }
  
  tau_ind_values <- numeric(100)  # Store tau_ind values from each simulation
  
  # Simulation loop
  for (sim in 1:100) {
    current_time <- 0
    statuses <- rep("available", n_listings)
    availability_times <- rep(0, n_listings)
    order_counts <- rep(0, n_listings)
    
    while (current_time < total_time) {
      # Update current time
      current_time <- current_time + time_step
      
      # Update statuses of all listings
      statuses <- update_statuses(statuses, availability_times, current_time)
      
      # Determine the number of customers arriving in this time step based on Poisson distribution
      n_customers_t1 <- rpois(1, lambda = lambda_tot/4 * time_step)
      n_customers_c1 <- rpois(1, lambda = lambda_tot/4 * time_step)
      n_customers_t2 <- rpois(1, lambda = lambda_tot/4 * time_step)
      n_customers_c2 <- rpois(1, lambda = lambda_tot/4 * time_step)
      
      available_indices <- which(statuses == "available")
      
      if (length(available_indices) > 0 && n_customers_t1 > 0) {
        for (j in 1:n_customers_t1) {
          if (length(available_indices) == 0) break
          C <- rep(0, n_listings)
          C[available_indices] <- rbinom(length(available_indices), 1, 0.5)
          consideration_indices <- which(C == 1)
          if (length(consideration_indices) > 0) {
            v <- ifelse(treatment_listings == "treatment", 0.17*1.25, 0.17)
            probabilities <- v[consideration_indices] / (1 + sum(v[consideration_indices]))
            ordered_index <- sample(c(consideration_indices, n_listings + 1), 1, prob = c(probabilities, 1 - sum(probabilities)))
            if (ordered_index != n_listings + 1) {
              statuses[ordered_index] <- "unavailable"
              availability_times[ordered_index] <- current_time + rexp(1, rate = 1)
              order_counts[ordered_index] <- order_counts[ordered_index] + 1
            }
          }
        }
      }
      
      if (length(available_indices) > 0 && n_customers_c1 > 0) {
        for (j in 1:n_customers_c1) {
          if (length(available_indices) == 0) break
          C <- rep(0, n_listings)
          C[available_indices] <- rbinom(length(available_indices), 1, 0.5)
          consideration_indices <- which(C == 1)
          if (length(consideration_indices) > 0) {
            v <- rep(0.17, n_listings)
            probabilities <- v[consideration_indices] / (1 + sum(v[consideration_indices]))
            ordered_index <- sample(c(consideration_indices, n_listings + 1), 1, prob = c(probabilities, 1 - sum(probabilities)))
            if (ordered_index != n_listings + 1) {
              statuses[ordered_index] <- "unavailable"
              availability_times[ordered_index] <- current_time + rexp(1, rate = 1)
              order_counts[ordered_index] <- order_counts[ordered_index] + 1
            }
          }
        }
      }
      if (length(available_indices) > 0 && n_customers_t2 > 0) {
        for (j in 1:n_customers_t2) {
          if (length(available_indices) == 0) break
          C <- rep(0, n_listings)
          C[available_indices] <- rbinom(length(available_indices), 1, 0.5)
          consideration_indices <- which(C == 1)
          if (length(consideration_indices) > 0) {
            v <- ifelse(treatment_listings == "treatment", 0.51*1.25, 0.51)
            probabilities <- v[consideration_indices] / (1 + sum(v[consideration_indices]))
            ordered_index <- sample(c(consideration_indices, n_listings + 1), 1, prob = c(probabilities, 1 - sum(probabilities)))
            if (ordered_index != n_listings + 1) {
              statuses[ordered_index] <- "unavailable"
              availability_times[ordered_index] <- current_time + rexp(1, rate = 1)
              order_counts[ordered_index] <- order_counts[ordered_index] + 1
            }
          }
        }
      }
      
      if (length(available_indices) > 0 && n_customers_c1 > 0) {
        for (j in 1:n_customers_c1) {
          if (length(available_indices) == 0) break
          C <- rep(0, n_listings)
          C[available_indices] <- rbinom(length(available_indices), 1, 0.5)
          consideration_indices <- which(C == 1)
          if (length(consideration_indices) > 0) {
            v <- rep(0.51, n_listings)
            probabilities <- v[consideration_indices] / (1 + sum(v[consideration_indices]))
            ordered_index <- sample(c(consideration_indices, n_listings + 1), 1, prob = c(probabilities, 1 - sum(probabilities)))
            if (ordered_index != n_listings + 1) {
              statuses[ordered_index] <- "unavailable"
              availability_times[ordered_index] <- current_time + rexp(1, rate = 1)
              order_counts[ordered_index] <- order_counts[ordered_index] + 1
            }
          }
        }
      }
    }
    
    # Calculate tau_ind for this simulation
    Y <- order_counts / total_time
    df <- data.frame(status = statuses, treatment = treatment_listings)
    N <- df %>% filter(status == "available") %>% nrow() - 1
    N<-rep(N, n_listings)
    M <- df %>% filter(status == "available", treatment == "treatment") %>% nrow()
    M<-rep(M, n_listings)
    M[treatment_listings == 'treatment'] <- M[treatment_listings == 'treatment'] - 1
    beta <- 2*sum(2*M-N)/n_listings/eig_1[1]
    tau_ind <- 1/n_listings * sum(order_counts/total_time * (M/0.5-(N-M)/0.5) + beta * eig_1[1]); tau_ind
    print(tau_ind)
    
    tau_ind_values[sim] <- tau_ind
  }
  
  return(tau_ind_values)
}

# Run the simulation
tau_ind_values <- run_simulation()

# Plot histogram of tau_ind values
hist(tau_ind_values/10000, breaks = 30, col = "skyblue", xlab = "tau_ind_pc", main = "Histogram of PC Balancing Estimator (Heterogeneous)")
hist(tau_ind_values/10000, breaks = 30, col = "skyblue", 
     xlab = "tau_ind_pc", 
     main = "Histogram of PC Balancing Estimator (Homogeneous)", 
     cex.main = 0.8)

